<?php
/**
 * NOTICE OF LICENSE
 *
 * This file is licenced under the GNU General Public License, version 3 (GPL-3.0).
 * With the purchase or the installation of the software in your application
 * you accept the licence agreement.
 *
 * @author    emarketing www.emarketing.com <integrations@emarketing.com>
 * @copyright 2022 emarketing AG
 * @license   https://opensource.org/licenses/GPL-3.0 GNU General Public License version 3
 */

namespace Emarketing\Service;

use Emarketing\Service\Snippets\Snippets;

/**
 * Class GoogleTracker
 * @package Emarketing\Service
 */
class GoogleTracker extends Snippets
{
    /**
     * @param $globalSiteTracker
     * @param $viewItemTracker
     * @param $addToCartTracker
     * @param $conversionTracker
     * @return bool
     * @throws \Exception
     */
    public function saveTracker($globalSiteTracker, $viewItemTracker, $addToCartTracker, $conversionTracker)
    {
        $this->saveInConfiguration('GLOBAL_SITE_TRACKER', $globalSiteTracker);
        $this->saveInConfiguration('VIEW_ITEM_TRACKER', $viewItemTracker);
        $this->saveInConfiguration('ADD_TO_CART_TRACKER', $addToCartTracker);
        $this->saveInConfiguration('CONVERSION_TRACKER', $conversionTracker);

        return true;
    }

    /**
     * @return string
     */
    public function getGlobalSiteTracker()
    {
        return $this->getFromConfiguration('GLOBAL_SITE_TRACKER');
    }

    /**
     * @return string
     */
    public function getViewItemTracker()
    {
        $tracker = $this->getFromConfiguration('VIEW_ITEM_TRACKER');

        if (empty($tracker)) {
            return '';
        }

        $tracker = str_replace(['<script>', '</script>'], ['', ''], $tracker);

        $script = "<script>\nvar emarketingGoogleViewItem = function() {" . $this->replaceProductPageVariables($tracker) . "};\n";
        $script .= "document.addEventListener('DOMContentLoaded', emarketingGoogleViewItem);\n";
        $script .= "document.addEventListener('DOMContentLoaded', function(event) { prestashop.on('updatedProduct', emarketingGoogleViewItem)});\n</script>";

        return $script;
    }

    /**
     * @return string
     */
    public function getAddToCartTracker()
    {
        $tracker = $this->getFromConfiguration('ADD_TO_CART_TRACKER');

        if (empty($tracker)) {
            return '';
        }

        $tracker = str_replace(['<script>', '</script>'], ['', ''], $tracker);

        $script = "<script>\nvar emarketingGoogleAdd2Cart = function() { document.querySelectorAll('.add-to-cart, #add_to_cart button, #add_to_cart a, #add_to_cart input').forEach(function(a){\na.addEventListener('click', function(){" . $this->replaceProductPageVariables($tracker) . "});});};\n";
        $script .= "document.addEventListener('DOMContentLoaded', emarketingGoogleAdd2Cart);\n";
        $script .= "document.addEventListener('DOMContentLoaded', function(event) { prestashop.on('updatedProduct', emarketingGoogleAdd2Cart)});\n</script>";

        return $script;
    }

    /**
     * @return string
     * @throws \Exception
     */
    public function getConversionTracker()
    {
        $conversionTracker = $this->getFromConfiguration('CONVERSION_TRACKER');

        $order = new \Order(\Tools::getValue('id_order'));

        $currency = new \Currency($order->id_currency);

        return preg_replace(
            ['/\'value\'\:[\s\S]+?,/', '/\'currency\'\:[\s\S]+?,/', '/\'transaction_id\'\:[\s\S]+?\'\'/'],
            ["'value': " . \Tools::ps_round($order->total_paid, 2) . ",", "'currency': '" . $currency->iso_code  . "',", "'transaction_id': '" . $order->id . "',"],
            $conversionTracker
        );
    }

    /**
     * @param $tracker
     * @return string
     */
    private function replaceProductPageVariables($tracker)
    {
        return preg_replace(
            ['/\'REPLACE_NAME\'/', '/\'REPLACE_CATEGORY\'/', '/\'REPLACE_ID\'/', '/\bREPLACE_VALUE\b/'],
            ['document.getElementById("emarketing_product_name").value', 'document.getElementById("emarketing_product_category").value', 'document.getElementById("emarketing_product_id").value', 'document.getElementById("emarketing_product_price").value'],
            $tracker
        );
    }
}
