<?php

namespace PrestaShop\Module\PulsePs\Service;

use GuzzleHttp\Client;
use GuzzleHttp\Exception\RequestException;
use Configuration;
use PrestaShop\Module\PulsePs\Service\Exception\PulseException;
use Validate;
use Link;

class Pulse {

    /** @var string */
    const CLIENT_ID = 'PULSE_CLIENT_ID';

    /** @var string */
    const ACCESS_CODE = 'PULSE_ACCESS_CODE';

    /** @var string */
    const CLIENT_SECRET = 'PULSE_CLIENT_SECRET';

    /** @var string */
    const ACCESS_TOKEN = 'PULSE_ACCESS_TOKEN';

    /** @var string */
    const ACCESS_TOKEN_EXPIRES_AT = 'PULSE_ACCESS_TOKEN_EXPIRES_AT';

    /** @var string */
    const REFRESH_TOKEN = 'PULSE_REFRESH_TOKEN';

    /** @var string */
    const REFRESH_TOKEN_EXPIRES_AT = 'PULSE_REFRESH_TOKEN_EXPIRES_AT';

    /** @var string */
    const GROUP_ID = 'PULSE_GROUP_ID';

    /** @var string */
    const HOSTNAME = 'PULSE_HOSTNAME';

    /** @var string */
    const REDIRECT_URI = 'PULSE_REDIRECT_URI';

    /** @var string */
    const SCOPES = 'PULSE_SCOPES';

    /** @var string */
    const SCOPES_VALUE = 'contacts.create contacts.read contacts.update contacts.delete contactgroups.read';

    /** @var string */
    const NOT_CONNECTED_WARNING = 'Not connected to Pulse. <br />Please complete the setup and connect to Pulse to use this module.';

    private $http;

    public function __construct(){
        $this->http = new Client([
            'base_url' => $this->getHostname(),
        ]);
    }

    private function makeRequest(string $method = 'GET', string $endpoint = '', array $params = [])
    {
        if(!$this->accessTokenValid()){
            try{
                $this->refreshToken();
            } catch (RequestException $e){
                throw new PulseException('Pulse Token expired. Please review the module settings to connect to Pulse.');
            }
            return $this->makeRequest($method, $endpoint, $params);
        }

        $endpoint = '/api/v1/' . $endpoint;
        $headers = ['Authorization' => 'Bearer ' . $this->getAccessToken()];
        try{
            switch ($method){
                case 'GET':
                    $response = $this->http->get($endpoint, [
                        'query' => $params,
                        'headers' => $headers
                    ]);
                    break;
                case 'POST':
                    $response = $this->http->post($endpoint, [
                        'body' => $params,
                        'headers' => $headers
                    ]);
                    break;
                case 'PUT':
                    $response = $this->http->put($endpoint, [
                        'body' => $params,
                        'headers' => $headers
                    ]);
                    break;
                case 'DELETE':
                    $response = $this->http->delete($endpoint, [
                        'headers' => $headers
                    ]);
                    break;
                default:
                    $response = false;
                    throw new \Exception('Method not available');
                    break;
            }
        } catch (RequestException $e){
            $response = $e->getResponse();
            if($response->getStatusCode() === 401){
                throw new PulseException($e->getMessage());
            }
            $results = json_decode($response->getBody()->getContents(), true);
            if($results['error']){
                throw new PulseException($results['error']['message']);
            } else {
                throw new PulseException($e->getMessage());
            }
        }

        return json_decode($response->getBody()->getContents(), true);
    }

    public function requestToken(string $authorization_code) :void
    {
        try {
            $response = $this->http->post('token', [
                'body' => [
                    'client_id' => $this->getClientId(),
                    'grant_type' => 'authorization_code',
                    'client_secret' => $this->getClientSecret(),
                    'redirect_uri' => $this->getRedirectUri(),
                    'code' => $authorization_code
                ]
            ]);
            $results = json_decode($response->getBody()->getContents(), true);

            $this->setAccessToken($results['access_token']);
            $this->setRefreshToken($results['refresh_token']);
            $this->setAccessTokenExpiresAt(\DateInterval::createFromDateString($results['expires_in'] . ' ' . 'seconds'));
        } catch (\Exception $e){
            throw new PulseException($e->getMessage());
        }
    }

    public function refreshToken()
    {
        try{
            $response = $this->http->post('token', [
                'body' => [
                    'grant_type' => 'refresh_token',
                    'refresh_token' => $this->getRefreshToken(),
                    'client_id' => $this->getClientId(),
                    'client_secret' => $this->getClientSecret(),
                ]
            ]);
            $results = json_decode($response->getBody()->getContents(), true);
            $this->setRefreshToken($results['refresh_token']);
            $this->setAccessToken($results['access_token']);
            $this->setAccessTokenExpiresAt(\DateInterval::createFromDateString($results['expires_in'] . ' ' . 'seconds'));
        } catch (RequestException $e){
            // If refresh request fails, clear old access_token and refresh_token. A new oauth flow is required.
            if($e->getResponse()->getStatusCode() === 401){
                $this->clearTokens();
                throw $e;
            }
        }

    }

    public function getContacts(array $params = []) :array
    {
        return $this->makeRequest('GET', 'contacts/', $params);
    }

    /**
     * @param string $email
     * @return mixed|null
     */
    public function getContactByEmail(string $email)
    {
        $result = $this->getContacts([
            'filters' => [
                ['email', 'eq', $email]
            ]
        ]);
        if((int) $result['data']['count'] === 1){
            return $result['data']['contacts'][0];
        }
        return null;
    }

    /**
     * @param string $email
     * @return mixed|null
     */
    public function getContactByLocalId(int $id)
    {
        $result = $this->getContacts([
            'filters' => [
                ['remote_id', 'eq', $id]
            ]
        ]);
        if((int) $result['data']['count'] === 1){
            return $result['data']['contacts'][0];
        }
        return null;
    }

    /**
     * @param int $id
     * @return array
     * @throws PulseException
     */
    public function getContact(int $id) :array
    {
        $result = $this->makeRequest('GET', 'contacts/' . $id);
        return $result['data']['contact'];
    }

    public function createContact(
        string $email,
        string $first_name = null,
        string $last_name = null,
        string $company_name = null,
        int $local_id = null,
        array $group_ids = null,
        bool $receive_emails = true
    ) :int
    {
        $params = array();

        if (isset($email)) $params['email'] = $email;
        if (isset($first_name)) $params['first_name'] = $first_name;
        if (isset($last_name)) $params['last_name'] = $last_name;
        if (isset($company_name)) $params['company_name'] = $company_name;
        if (isset($local_id)) $params['remote_id'] = $local_id;
        if (isset($group_ids)) $params['group_ids'] = $group_ids;
        $params['receive_emails'] = $receive_emails;

        $results = $this->makeRequest('POST', 'contacts', $params);

        return $results['data']['id'];
    }

    public function deleteContact(int $id) :void
    {
        $this->makeRequest('DELETE', 'contacts/' . $id);
    }

    public function updateContact(
        int $id,
        string $email,
        string $first_name = null,
        string $last_name = null,
        string $company_name = null,
        int $local_id = null,
        array $group_ids = null,
        bool $receive_emails = true
    ) :void
    {
        $params = array();

        if (isset($email)) $params['email'] = $email;
        if (isset($first_name)) $params['first_name'] = $first_name;
        if (isset($last_name)) $params['last_name'] = $last_name;
        if (isset($company_name)) $params['company_name'] = $company_name;
        if (isset($local_id)) $params['remote_id'] = $local_id;
        if (isset($group_ids)) $params['group_ids'] = $group_ids;
        $params['receive_emails'] = $receive_emails;

        $this->makeRequest('PUT', 'contacts/' . $id, $params);
    }

    /**
     * @return array
     * @throws PulseException
     */
    public function getContactGroups() :array
    {
        return $this->makeRequest('GET', 'groups/');
    }

    /**
     * @return bool
     */
    public function pulseAuthExists() :bool
    {
        if (!$this->accessTokenExists() || !$this->refreshTokenExists()){
            return false;
        } else {
            return true;
        }
    }

    /**
     * @return bool
     */
    public function accessTokenExists() :bool
    {
        return (!empty($this->getAccessToken()));
    }

    public function accessTokenValid() :bool
    {
        return ($this->getAccessTokenExpiresAt() > (new \DateTime()));
    }

    /**
     * @return string
     */
    public function getAccessToken() :string
    {
        return Configuration::get(self::ACCESS_TOKEN, null, null, null, '');
    }

    /**
     * @return \DateTime
     * @throws \Exception
     */
    public function getAccessTokenExpiresAt() :\DateTime
    {
        return \DateTime::createFromFormat(
            'Y-m-d H:i:s',
            Configuration::get(self::ACCESS_TOKEN_EXPIRES_AT, null, null, null, (new \DateTime())->format('Y-m-d H:i:s'))
        );
    }

    /**
     * @param \DateInterval $dateInterval
     * @throws \Exception
     */
    public function setAccessTokenExpiresAt(\DateInterval $dateInterval) :void
    {
        $expires_at = (new \DateTime())->add($dateInterval);
        Configuration::updateValue(self::ACCESS_TOKEN_EXPIRES_AT, $expires_at->format('Y-m-d H:i:s'));
    }

    /**
     * @param string $access_token
     */
    public function setAccessToken(string $access_token) :void
    {
        Configuration::updateValue(self::ACCESS_TOKEN, $access_token);
    }

    /**
     * @return bool
     */
    public function refreshTokenExists() :bool
    {
        return (!empty($this->getRefreshToken()));
    }

    /**
     * @return string
     */
    public function getRefreshToken() :string
    {
        return Configuration::get(self::REFRESH_TOKEN, null, null, null, '');
    }

    /**
     * @param string $refresh_token
     */
    public function setRefreshToken(string $refresh_token) :void
    {
        Configuration::updateValue(self::REFRESH_TOKEN, $refresh_token);
    }

    public function clearTokens() :void
    {
        Configuration::deleteByName(self::ACCESS_TOKEN);
        Configuration::deleteByName(self::REFRESH_TOKEN);
        Configuration::deleteByName(self::ACCESS_TOKEN_EXPIRES_AT);
    }

    /**
     * @return bool
     */
    public function clientIdExists() :bool
    {
        return (!empty($this->getClientId()));
    }

    /**
     * @return string
     */
    public function getClientId() :string
    {
        return Configuration::get(self::CLIENT_ID, null, null, null, '');
    }

    /**
     * @param string $client_id
     */
    public function setClientId(string $client_id) :void
    {
        Configuration::updateValue(self::CLIENT_ID, $client_id);
    }

    /**
     * @return bool
     */
    public function clientSecretExists() :bool
    {
        return (!empty($this->getClientSecret()));
    }

    /**
     * @return string
     */
    public function getClientSecret() :string
    {
        return Configuration::get(self::CLIENT_SECRET, null, null, null, '');
    }

    /**
     * @param string $client_secret
     */
    public function setClientSecret(string $client_secret) :void
    {
        Configuration::updateValue(self::CLIENT_SECRET, $client_secret);
    }

    /**
     * @return bool
     */
    public function hostnameExists() :bool
    {
        return (!empty($this->getHostname()));
    }

    /**
     * @return string
     */
    public function getHostname() :string
    {
        return Configuration::get(self::HOSTNAME, null, null, null, '');
    }

    /**
     * @param string $hostname
     * @throws \Exception
     */
    public function setHostname(string $hostname) :void
    {
        if(!Validate::isAbsoluteUrl($hostname)){
            throw new \Exception('Invalid hostname');
        }
        Configuration::updateValue(self::HOSTNAME, $hostname);
    }

    /**
     * @return string
     */
    public function getRedirectUri() :string
    {
        return _PS_BASE_URL_ . (new Link)->getAdminLink('PulseOauthCallback');
    }

    /**
     * @return bool
     */
    public function groupIdExists() :bool
    {
        return (!empty($this->getGroupId()));
    }

    /**
     * @return int
     */
    public function getGroupId()
    {
        return Configuration::get(self::GROUP_ID, null, null, null, null);
    }

    /**
     * @param int $group_id
     */
    public function setGroupId(int $group_id) :void
    {
        Configuration::updateValue(self::GROUP_ID, $group_id);
    }

}